<?php

if (!defined('ABSPATH')) {
    exit;
}

function finmo_checkout_webhook_update() {
    http_response_code(200); // Always respond 200 to acknowledge webhook

    try {
        $raw = file_get_contents("php://input");
        $response = json_decode($raw, true);

        finmo_log("[Checkout Webhook] RAW Payload: " . $raw, 'debug');
        error_log("[Checkout Webhook] RAW Payload: " . $raw); // PHP native log

        $method = $_SERVER['REQUEST_METHOD'];
         // For POST/PATCH: check Finmo event type first
        $order_id = null;
        $checkout_id = null;
        $status = null;
        $is_refundable = null;
        $is_partially_refundable = null;
        if ($method === 'GET') {
            // Browser redirect (query params)
            if (isset($_GET['order_id'], $_GET['checkout_id'], $_GET['status'])) {
                $order_id    = (int) sanitize_text_field($_GET['order_id']);
                $checkout_id = (string) sanitize_text_field($_GET['checkout_id']);
                $status      = (string) sanitize_text_field($_GET['status']);
            }
        } else {
            // Webhook POST/PATCH payload

            if (is_array($response)) {
                // Checkout ID and status may be top-level or inside event_detail
                if (isset($response['checkout_id'])) {
                    $checkout_id = (string) $response['checkout_id'];
                } elseif (isset($response['event_detail']['checkout_id'])) {
                    $checkout_id = (string) $response['event_detail']['checkout_id'];
                }

                if (isset($response['status'])) {
                    $status = (string) $response['status'];
                } elseif (isset($response['event_detail']['status'])) {
                    $status = (string) $response['event_detail']['status'];
                }

                // Order ID may be in multiple places
                if (isset($response['order_id'])) {
                    $order_id = (int) $response['order_id'];
                } elseif (isset($response['metadata']['plugin_req_body']['order_id'])) {
                    $order_id = (int) $response['metadata']['plugin_req_body']['order_id'];
                } elseif (isset($response['event_detail']['metadata']['plugin_req_body']['order_id'])) {
                    $order_id = (int) $response['event_detail']['metadata']['plugin_req_body']['order_id'];
                } elseif (isset($response['event_detail']['organization_reference_id'])) {
                    // Finmo also sends order_id as organization_reference_id sometimes
                    $order_id = (int) $response['event_detail']['organization_reference_id'];
                }
            }
        }

        // If still missing any, log and return
        if (empty($order_id) || empty($checkout_id) || empty($status)) {
            finmo_log("[Checkout Webhook] Missing required parameters. order_id={$order_id}, checkout_id={$checkout_id}, status={$status}", 'error');
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            finmo_log("[Checkout Webhook] Order not found: {$order_id}", 'error');
            return;
        }

        if ($order->get_payment_method() !== "finmo_payments") {
            finmo_log("[Checkout Webhook] Ignored non-Finmo order {$order_id}", 'info');
            return;
        }

        $stored_checkout_id = get_post_meta($order_id, 'finmo_checkout_id', true);
        if ($stored_checkout_id != $checkout_id) {
            finmo_log("[Checkout Webhook] Checkout ID mismatch for order {$order_id}", 'error');
            return;
        }

        // Update order meta and add note
        if (isset($response['is_refundable'])) {
            $is_refundable = (bool) $response['is_refundable'];
        } elseif (isset($response['event_detail']['is_refundable'])) {
            $is_refundable = (bool) $response['event_detail']['is_refundable'];
        }

        if (isset($response['is_partially_refundable'])) {
            $is_partially_refundable = (bool) $response['is_partially_refundable'];
            } elseif (isset($response['event_detail']['is_partially_refundable'])) {
            $is_partially_refundable = (bool) $response['event_detail']['is_partially_refundable'];
        }
        if (!is_null($is_refundable)) {
                update_post_meta($order_id, '_finmo_is_refundable', $is_refundable ? 'yes' : 'no');
        }
        if (!is_null($is_partially_refundable)) {
            update_post_meta($order_id, '_finmo_is_partially_refundable', $is_partially_refundable ? 'yes' : 'no');
        }
        update_post_meta($order_id, '_finmo_checkout_status', $status);
        $order->add_order_note("Payment Webhook Received with Status {$status}");
        finmo_log("[Checkout Webhook] Order {$order_id} updated with status {$status}", 'info');

        // Handle WooCommerce order status
        finmo_handle_checkout_status($order, $order->get_status(), $status);

        // Only redirect if GET (browser redirect)
        if ($method === 'GET') {
            if (in_array($status, ['COMPLETED', 'AWAITING_FUNDS', 'PENDING'], true)) {
                wp_safe_redirect($order->get_checkout_order_received_url());
                exit;
            }
            wp_safe_redirect(wc_get_checkout_url());
            exit;
        }

    } catch (Exception $e) {
        finmo_log('[Checkout Webhook Exception] ' . $e->getMessage(), 'error');
        return;
    }
}

/**
 * Handle checkout status updates.
 */
function finmo_handle_checkout_status($order, $order_status, $checkout_status)
{
    switch ($checkout_status) {
        case "COMPLETED":
            // Instead of 'completed', set to 'processing'
            if (!in_array($order_status, ["processing", "failed"], true)) {
                $order->update_status("processing", __('Payment Completed'));
            }
            break;
        case "AWAITING_FUNDS":
            if ($order_status !== "on-hold") {
                $order->update_status("on-hold", 'Payment awaiting funds.');
            }
            break;
        case "PENDING":
            $order->update_status("pending", 'Payment is pending.');
            break;
        case "CANCELLED":
        case "FAILED":
        case "EXPIRED":
            $order->update_status("failed", __('Finmo Payment ') . ucfirst(strtolower($checkout_status)) . '.');
            break;
        default:
            finmo_log("[Checkout Webhook] Unknown checkout status {$checkout_status} for order {$order->get_id()}", 'warning');
            break;
    }
}

/**
 * Find WooCommerce order by Finmo checkout id.
 */
function finmo_find_order_by_checkout_id($checkout_id)
{
    if (empty($checkout_id)) {
        return null;
    }

    $args = [
        'numberposts' => 1,
        'post_type'   => 'shop_order',
        'post_status' => 'any',
        'meta_key'    => 'finmo_checkout_id',
        'meta_value'  => sanitize_text_field($checkout_id),
        'fields'      => 'ids',
    ];

    $posts = get_posts($args);
    return !empty($posts) ? wc_get_order((int) $posts[0]) : null;
}

function finmo_refund_webhook_update()
{
    http_response_code(200);
    try {
        $raw = file_get_contents("php://input");
        $response = json_decode($raw, true);

        finmo_log("[Refund Webhook] RAW Payload: " . $raw, 'debug');

        $order_id = null;
        $refund_id = null;
        $refund_status = null;

        if (isset($_GET['order_id'], $_GET['refund_id'], $_GET['status'])) {
            $order_id = (int) sanitize_text_field($_GET['order_id']);
            $refund_id = (string) sanitize_text_field($_GET['refund_id']);
            $refund_status = (string) sanitize_text_field($_GET['status']);
        }

        else {
		    finmo_log("Inside post of _separate_refund_webhook");
		    $order_id  = (int)$response['event_detail']['metadata']['plugin_req_body']['order_id'];
            $refund_id = (string)$response['event_detail']['refund_id'];
		    $refund_status = (string)$response['event_detail']['status'];
            finmo_log("order_id {$order_id}. ");
        }
        if (empty($order_id) || empty($refund_id) || empty($refund_status)) {
		    error_log("order_id={$order_id}, refund_id={$refund_id}, refund_status={$refund_status}");
            finmo_log('[Refund Webhook] Missing required parameters Inside Normal refund_webhook (POST/PATCH).', 'error');
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order || $order->get_payment_method() !== "finmo_payments") {
            finmo_log("[Refund Webhook] Invalid or non-Finmo order: {$order_id}", 'error');
            return;
        }

        // Check refund_id matches stored meta
        $finmo_refund_id = get_post_meta($order_id, 'finmo_refund_id', true);
        if (!empty($finmo_refund_id) && $finmo_refund_id != $refund_id) {
            finmo_log("[Refund Webhook] Refund ID mismatch for order {$order_id}", 'error');
            return;
        }

        update_post_meta($order_id, '_finmo_refund_status', $refund_status);
        $order->add_order_note("Refund Webhook Received with Status {$refund_status}");
	    $status=$order->get_status();
	    finmo_log("status {$status}. refund status {$refund_status}");
        finmo_handle_refund_status($order, $order->get_status(), $refund_status);

        finmo_log("[Refund Webhook] Processed refund status {$refund_status} for order {$order_id}", 'info');
    }
    catch (Exception $e) {
        finmo_log('[Refund Webhook Exception] ' . $e->getMessage(), 'error');
        return;
    }

}
/**
 * Handle refund status updates.
 */
function finmo_handle_refund_status($order, $order_status, $refund_status)
{
    if (in_array($order_status, ["refunded", "completed", "on-hold"], true)) {
        switch ($refund_status) {
            case "COMPLETED":
                $order->update_status('refunded', __('Finmo Payment refunded.', 'finmo-payments'));
                $order->add_order_note('Finmo Refund completed.');
                break;
		    case "PENDING" :
                $order->update_status('on-hold',__('Finmo Refund is Pending.', 'finmo-payments'));
                $order->add_order_note('Finmo Refund Pending.');
                break;
            case "FAILED":
                $order->add_order_note('Finmo Refund failed.');
                break;
            default:
                $order->add_order_note("Unknown refund status: {$refund_status}");
                finmo_log("[Refund Webhook] Unknown refund status {$refund_status} for order {$order->get_id()}", 'warning');
                break;
        }
    } else {
        finmo_log("[Refund Webhook] Ignored refund update for order {$order->get_id()} with status {$order_status}", 'info');
    }
}